///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//  Copyright  NetworkDLS 2002, All rights reserved
//
// THIS CODE AND INFORMATION IS PROVIDED "AS IS" WITHOUT WARRANTY OF 
// ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO 
// THE IMPLIED WARRANTIES OF MERCHANTABILITY AND/OR FITNESS FOR A 
// PARTICULAR PURPOSE.
///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

#ifndef _Command_CPP
#define _Command_CPP
///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

#include <Windows.H>
#include <WindowsX.H>
#include <ShellAPI.H>
#include <Stdio.H>
#include <Stdlib.H>

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

#include "../Resources/Resource.H"

#include "../../SharedSource/LZARICode.H"
#include "../../SharedSource/RLECompression.H"
#include "../../SharedSource/ZLibCompression.H"

#include "../../SharedClasses/SQLClass/cRecordSet.H"
#include "../../SharedClasses/SQLClass/cSQL.H"
#include "../../SharedClasses/LZSS/LZSS_Compress.H"
#include "../../SharedClasses/LZSS/LZSS_Uncompress.H"

#include "../CSockSrvr/CSockSrvr.H"

#include "../../SharedSource/Debug.H"
#include "../../SharedSource/NSWFL.H"
#include "../../SharedSource/Common.H"
#include "../../SharedSource/CRC32.H"

#include "Init.H"
#include "Routines.H"
#include "Command.H"
#include "SQLExport.H"
#include "Console.H"
#include "DBInit.H"

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

bool ConnectToServer(bool bForcedConnect)
{
	char sIPAddress[64];
	char sSQL[1024];
	CRecordSet rsTemp;
	
	if(!IsClientConfigured(false))
	{
		WriteSysLogEx("Cannot connect to the SQL-Exchange server: Not configured.", EVENT_WARN);
		return false;
	}

	if(gServer.bcConnected[CCI.iThisClient])
	{
		WriteSysLogEx("Failed to connect to the SQL-Exchange server: Already connected.", EVENT_WARN);
		return false;
	}

	if(!GetIPAddress(sIPAddress, sizeof(sIPAddress), gsServerAddress))
    {
		WriteSysLogEx("Failed to connect to the SQL-Exchange server: Error resolving hostname.", EVENT_ERROR);
        return false;
    }

	if(CCI.cCustSQL.Connect(gsSQLDriver, gsSQLServer, gsSQLUserID, gsSQLPassword, "Master", gsDBMode))
	{
		WriteSysLogEx("Successfully connected to the SQL Master database.", EVENT_NONE);
		CCI.bCustDBConnected = true;
	}
	else{
		WriteSysLogEx("Failed to connected to the SQL Master database.", EVENT_ERROR);
		return false;						
	}

	if(!bForcedConnect)
	{
		sprintf_s(sSQL, sizeof(sSQL), "SELECT Count([TransTable])"
			" FROM [%s].[%s].SQLExch_Trans",
			gsReplicationDB, gsDefaultDBO);

		if(!CCI.cCustSQL.Execute(sSQL, &rsTemp))
		{
			WriteSysLogEx("Failed to get the pending transaction count.", EVENT_ERROR);

			rsTemp.Close();
			CCI.cCustSQL.Disconnect();
			return false;
		}

		if(rsTemp.RowCount <= 0)
		{
			WriteSysLogEx("No new transactions are available, no need to connect.", EVENT_NONE);
			rsTemp.Close();
			CCI.cCustSQL.Disconnect();
			return true;
		}
		else{
			if(rsTemp.Fetch())
			{
				if(rsTemp.lColumn(1) <= 0)
				{
					WriteSysLogEx("No new transactions are available, no need to connect.", EVENT_NONE);
					rsTemp.Close();
					CCI.cCustSQL.Disconnect();
					return true;
				}
				else{
					//All is well, we can connect.
				}
			}
			else{
				WriteSysLogEx("No new transactions are available, no need to connect.", EVENT_NONE);
				rsTemp.Close();
				CCI.cCustSQL.Disconnect();
				return true;
			}
		}
	}

	if(gServer.DoConnect(sIPAddress, gdwServerPort, &CCI.iThisClient))
	{
		WriteSysLogEx("Connected successfully to the SQL-Exchange server.", EVENT_NONE);
	}
	else{
		WriteSysLogEx("Failed to connect to the SQL-Exchange server.", EVENT_WARN);

		if(gServer.bcConnected[CCI.iThisClient])
		{
			WriteSysLogEx("Dropping a possible locked connection.", EVENT_WARN);
			gServer.bcDisconnect[CCI.iThisClient] = true;
		}

		CCI.cCustSQL.Disconnect();
		return false;
	}

	return true;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/*
	int ProcessCommand(CSockSrvr *pSockSrvr, int iClient, char *sRecvBuf, int iRecvBufSz);

	Processes a command from the peer.

	Possible return Values:
		CMD_OK    The command was a success, all is well.
		CMD_DONE  The command was a success, close the connection.
		CMD_ERROR The command failed due to an error.
*/
int ProcessCommand(CSockSrvr *pSockSrvr, int iClient, char *sCmdBuf, int iCmdBufSz)
{
    char sCmdData[SENDBUFSZ + RECVBUFSZ];
	char sTemp[2048];

	int iSendBufSz = 0;
	int iCmdDataSz = 0;
	int iCmdFlagLength = 0;

	//------------------------------------------------------------------------------------------------
	//The "::DumpTable" command is sent by the server when a corrupt 
	//	file is received durring the database initialization process.
	//
	//	Before this command can be executed, the server must first send the following commands.
	//		"::TransDataDatabase->"  - The name of the database for the table dump.
	//		"::TransDataTable->"     - The name of the table to dump.
	//		"::TransDataDBO->"       - The database owner for the table dump.
	//
	if((iCmdFlagLength = CmdCmp(sCmdBuf, "::DumpTable")))
    {
		if(gbDebugMode)
		{
			WriteLogEx(pSockSrvr->icClientID[iClient], "Received dump table command.", EVENT_NONE);
		}

		iCmdDataSz = BreakCmdFromData(sCmdBuf, iCmdFlagLength, iCmdBufSz, sCmdData);

		char sTempFile[MAX_PATH];

		GetTempFileName(sTempFilesPath, "SEC", 0, sTempFile);

		sprintf_s(sTemp, sizeof(sTemp), "SELECT * FROM [%s].[%s].[%s]",
			CCI.sTransDatabase,
			CCI.sTransDBO,
			CCI.sTransTable);
		
		int iSQLExpRes = ExportSQLResults(pSockSrvr, iClient, 0, sTemp,
			CCI.sTransDatabase, CCI.sTransDBO, CCI.sTransTable,
			sTempFile);

		if(!CCI.cCustSQL.ExecuteNonQuery("USE [Master]"))
		{
			WriteLogEx(pSockSrvr->icClientID[iClient], "Failed to change the mster database context. Performance will be affected.", EVENT_WARN);
		}

		if(iSQLExpRes == SQL_IMPORT_RESULT_ZEROROWS)
		{
			DeleteFile(sTempFile);
			return CMD_OK;
		}
		else if(iSQLExpRes == SQL_IMPORT_RESULT_OK){

			sprintf_s(sTemp, RECVBUFSZ, "::TransDataFileSize->%d", Get_FileSize(sTempFile));
			pSockSrvr->SetNextSendData(iClient, sTemp);

			sprintf_s(sTemp, RECVBUFSZ, "::TransDataDatabase->%s", CCI.sTransDatabase);
			pSockSrvr->SetNextSendData(iClient, sTemp);

			sprintf_s(sTemp, RECVBUFSZ, "::TransDataTable->%s", CCI.sTransTable);
			pSockSrvr->SetNextSendData(iClient, sTemp);

			sprintf_s(sTemp, RECVBUFSZ, "::TransDataDBO->%s", CCI.sTransDBO);
			pSockSrvr->SetNextSendData(iClient, sTemp);

			pSockSrvr->SetNextSendData(iClient, "::SendingTransData");

			SendFileData(pSockSrvr, iClient, sTempFile, "TransDumpCRC");
			return CMD_OK;
		}
		else if(iSQLExpRes == SQL_IMPORT_RESULT_ERROR)
		{
			DeleteFile(sTempFile);
			return CMD_ERROR;
		}

		return CMD_OK;
	}
	//------------------------------------------------------------------------------------------------
	//Receive the Transaction Data File database.
	else if((iCmdFlagLength = CmdCmp(sCmdBuf, "::TransDataDatabase->")))
    {
		iCmdDataSz = BreakCmdFromData(sCmdBuf, iCmdFlagLength, iCmdBufSz, sCmdData);
		strcpy_s(CCI.sTransDatabase, sizeof(CCI.sTransDatabase), sCmdData);
		return CMD_OK;
	}
	//------------------------------------------------------------------------------------------------
	//Receive the Transaction Data File table.
	else if((iCmdFlagLength = CmdCmp(sCmdBuf, "::TransDataTable->")))
    {
		iCmdDataSz = BreakCmdFromData(sCmdBuf, iCmdFlagLength, iCmdBufSz, sCmdData);
		strcpy_s(CCI.sTransTable, sizeof(CCI.sTransTable), sCmdData);
		return CMD_OK;
	}
	//------------------------------------------------------------------------------------------------
	//Receive the Transaction Data File default DBO.
	else if((iCmdFlagLength = CmdCmp(sCmdBuf, "::TransDataDBO->")))
    {
		iCmdDataSz = BreakCmdFromData(sCmdBuf, iCmdFlagLength, iCmdBufSz, sCmdData);
		strcpy_s(CCI.sTransDBO, sizeof(CCI.sTransDBO), sCmdData);
		return CMD_OK;
	}
	//------------------------------------------------------------------------------------------------
	//The "::BeginInit" command is sent when the entire client database(s)
	//	need to be synchronized with the server.
	else if((iCmdFlagLength = CmdCmp(sCmdBuf, "::BeginInit->")))
    {
		if(gbDebugMode)
		{
			WriteLogEx(pSockSrvr->icClientID[iClient], "Received the begin init.", EVENT_NONE);
		}

		char sTempFile[MAX_PATH];

		iCmdDataSz = BreakCmdFromData(sCmdBuf, iCmdFlagLength, iCmdBufSz, sCmdData);
		CCI.iInitStep = atol(sCmdData);

		WriteLogEx(pSockSrvr->icClientID[iClient], "Server is requesting that the replication be initialized.", EVENT_INFO);

		if(CCI.iInitStep == 0)
		{
			GetTempFileName(sTempFilesPath, "SEC", 0, sTempFile);

			if(GenerateDBCreationSQL(&CCI.cCustSQL, pSockSrvr, iClient, sTempFile))
			{
				pSockSrvr->SetNextSendData(iClient, "::SendingDBCreationSQL");
				SendFileData(pSockSrvr, iClient, sTempFile, "DBCreationCRC");
			}
			else{
				//FIXME: Should be an error.				
			}

			if(!CCI.cCustSQL.ExecuteNonQuery("USE [Master]"))
			{
				WriteLogEx(pSockSrvr->icClientID[iClient], "Failed to change Master database context. Performance will be affected.", EVENT_WARN);
			}

			DeleteFile(sTempFile);
		}
		else{
			pSockSrvr->SetNextSendData(iClient, "::ResumeInit");
		}

		return CMD_OK;
	}
	//------------------------------------------------------------------------------------------------
	//The "::BeginReplication->" is sent when the server is ready to reveice
	//	 only pending synchronization transactions.
	else if((iCmdFlagLength = CmdCmp(sCmdBuf, "::BeginReplication->")))
    {
		if(gbDebugMode)
		{
			WriteLogEx(pSockSrvr->icClientID[iClient], "Received the begin transaction.", EVENT_NONE);
		}

		iCmdDataSz = BreakCmdFromData(sCmdBuf, iCmdFlagLength, iCmdBufSz, sCmdData);

		if( _strcmpi(sCmdData, "Transactions") == 0)
		{
			//Need to set the "SQLExch_Trans.SQLExch_Pending" flag to 1
			//	so that we dont delete any new transaction when we clear
			//	the table after the replication process.
			sprintf_s(sTemp, sizeof(sTemp),
				"UPDATE [%s].[%s].SQLExch_Trans"
				" SET SQLExch_Pending = 1",
				gsReplicationDB, gsDefaultDBO);
			CCI.cCustSQL.ExecuteNonQuery(sTemp);

			//Get a list of all of the statements that need to be ran.
			sprintf_s(sTemp, sizeof(sTemp),
				"SELECT DISTINCT NULL AS [ID], [A].[RepTable], [A].[Statement], [A].[OnSuccess],"
				" [A].[OnFailure], [A].[TransDB], [A].[ImportTable], [A].[Sequence]"
				" FROM [%s].[%s].[SQLExch_Statements] AS [A]"
				" INNER JOIN [%s].[%s].[SQLExch_Trans] AS [B] ON [A].[TransDB] = [B].[TransDB] AND [B].[TransTable] = [A].[TransTable]"
				" WHERE [A].[Active] = 1 AND [B].[SQLExch_Pending] = 1"
				" ORDER BY [A].[Sequence], [A].[Statement], [A].[OnSuccess], [A].[OnFailure], [A].[ImportTable]",
				gsReplicationDB, gsDefaultDBO,
				gsReplicationDB, gsDefaultDBO);
		}
		else if( _strcmpi(sCmdData, "All") == 0)
		{
			//Get a list of all of the statements that need to be ran.
			sprintf_s(sTemp, sizeof(sTemp),
				"SELECT [ID], NULL AS [RepTable], 'SELECT * FROM"
				" [' + TransDB + '].[%s].[' + TransTable + ']',"
				" NULL AS [OnSuccess], NULL AS [OnFailure], [TransDB], [TransTable]"
				" FROM [%s].[%s].[SQLExch_Statements]"
				" WHERE [PrimarySelect] = 1"
				" ORDER BY [ID] ASC",
				gsDefaultDBO, gsReplicationDB, gsDefaultDBO);
		}
		else if( _strcmpi(sCmdData, "Resume") == 0)
		{
			//Get a list of all of the statements that need to be ran.
			sprintf_s(sTemp, sizeof(sTemp),
				"SELECT [ID], NULL AS [RepTable], 'SELECT * FROM"
				" [' + TransDB + '].[%s].[' + TransTable + ']',"
				" NULL AS [OnSuccess], NULL AS [OnFailure], [TransDB], [TransTable]"
				" FROM [%s].[%s].[SQLExch_Statements]"
				" WHERE [PrimarySelect] = 1"
				" AND [ID] > %d"
				" ORDER BY [ID] ASC",
				gsDefaultDBO, gsReplicationDB, gsDefaultDBO, CCI.iInitStep);
		}
		CCI.rsTrans.bReplaceSingleQuotes = false;
		if(!CCI.cCustSQL.Execute(sTemp, &CCI.rsTrans))
		{
			WriteLogEx(pSockSrvr->icClientID[iClient], "Failed to get a list of pending transactions.", EVENT_ERROR);
			return CMD_ERROR;
		}

		sprintf_s(sTemp, sizeof(sTemp), "Processing %d replication statements.", CCI.rsTrans.RowCount);
		WriteLogEx(pSockSrvr->icClientID[iClient], sTemp, EVENT_NONE);

		return ExecuteTransactionSQL(pSockSrvr, iClient);
	}
	//------------------------------------------------------------------------------------------------
	//The server is letting us know what compression level to use, if any.
	else if((iCmdFlagLength = CmdCmp(sCmdBuf, "::CompressionLevel->")))
    {
		if(gbDebugMode)
		{
			WriteLogEx(pSockSrvr->icClientID[iClient], "Received compression level option.", EVENT_NONE);
		}

		iCmdDataSz = BreakCmdFromData(sCmdBuf, iCmdFlagLength, iCmdBufSz, sCmdData);

		CCI.dwCompressionLevel = atol(sCmdData);

		if(CCI.dwCompressionLevel < 0)
		{
			CCI.dwCompressionLevel = 0;
		}

		return CMD_OK;
	}
	//------------------------------------------------------------------------------------------------
	//The server is letting us know what compression algorithm to use, if any.
	else if((iCmdFlagLength = CmdCmp(sCmdBuf, "::CompressionMethod->")))
    {
		if(gbDebugMode)
		{
			WriteLogEx(pSockSrvr->icClientID[iClient], "Received compression option.", EVENT_NONE);
		}

		iCmdDataSz = BreakCmdFromData(sCmdBuf, iCmdFlagLength, iCmdBufSz, sCmdData);

		if(strlen(sCmdData) > 0 && _strcmpi(sCmdData, "None") != 0)
		{
			strcpy_s(CCI.sCompressionMethod, sizeof(CCI.sCompressionMethod), sCmdData);
			CCI.bUseCompression = true;
		}
		else{
			WriteLogEx(pSockSrvr->icClientID[iClient], "Compression is off. This may overwork the network.", EVENT_WARN);
			CCI.bUseCompression = false;
		}

		return CMD_OK;
	}
	//------------------------------------------------------------------------------------------------
	//This is a general processing message from the server.
	//	All messages are file logged and written to the event log.
	else if((iCmdFlagLength = CmdCmp(sCmdBuf, "::Msg->")))
    {
		if(gbDebugMode)
		{
			WriteLogEx(pSockSrvr->icClientID[iClient], "Received a message from the server.", EVENT_NONE);
		}

		iCmdDataSz = BreakCmdFromData(sCmdBuf, iCmdFlagLength, iCmdBufSz, sCmdData);
		WriteLogEx(pSockSrvr->icClientID[iClient], sCmdData, EVENT_INFO);

		return CMD_OK;
	}
	//------------------------------------------------------------------------------------------------
    //The "::LastImportSuccess" message is sent by the server when the last
	//	data file sent to the server was successfully imported.
	//Run the On Success SQL scripts.	
	else if((iCmdFlagLength = CmdCmp(sCmdBuf, "::LastImportSuccess")))
    {
		//The last import succeeded. If we need to take action then this is where to do it.
		WriteLogEx(pSockSrvr->icClientID[iClient], "Last import was a Success.", EVENT_NONE);

		bool bResult = false;

		if(strlen(CCI.sOnSuccess) > 0)
		{
			bResult = CCI.cCustSQL.ExecuteNonQuery(CCI.sOnSuccess);
			strcpy_s(CCI.sOnSuccess, sizeof(CCI.sOnSuccess), "");
			if(!bResult)
			{
				WriteLogEx(pSockSrvr->icClientID[iClient], "Failed to execute the OnSuccess statement.", EVENT_ERROR);
				return CMD_ERROR;
			}
		}

		//CCI.cCustSQL.ExecuteNonQuery(sCmdData); FIXME: What the hell is this for?

		return ExecuteTransactionSQL(pSockSrvr, iClient);
	}
	//------------------------------------------------------------------------------------------------
    //The "::LastImportFailed" message is sent by the server when the last
	//	data file sent to the server failed to import properly for any reason.
	//Run the On Failure SQL scripts.	
	else if((iCmdFlagLength = CmdCmp(sCmdBuf, "::LastImportFailed")))
    {
		//The last import failed. If we need to take action then this is where to do it.

		bool bResult = false;

		WriteLogEx(pSockSrvr->icClientID[iClient], "Last SQL import failed.", EVENT_ERROR);

		if(strlen(CCI.sOnFailure) > 0)
		{
			bResult = CCI.cCustSQL.ExecuteNonQuery(CCI.sOnFailure);
			strcpy_s(CCI.sOnFailure, sizeof(CCI.sOnFailure), "");
			if(!bResult)
			{
				WriteLogEx(pSockSrvr->icClientID[iClient], "Failed to execute the OnFailure statement.", EVENT_ERROR);
				return CMD_ERROR;
			}
		}

		if(ExecuteTransactionSQL(pSockSrvr, iClient))
		{
			return CMD_OK;
		}
		else{
			return CMD_ERROR;
		}

		return CMD_OK;
	}
	//------------------------------------------------------------------------------------------------

	WriteLogEx(pSockSrvr->icClientID[iClient], "Received unknown command.", EVENT_ERROR);

	return CMD_ERROR;	
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

bool SendFileData(CSockSrvr *pSockSrvr, const int iClient, const char *sFileName, const char *sCRCTag)
{
    FILE *hSourceHandle = NULL;

	char sFullFile[MAX_PATH];
    char sFileBuffer[FILEBUFFERSZ + 1];
	int iBytesRead = 0;

	bool bSendCRC = false;

    DWORD dwDataCRC = 0xffffffff;

	if(strlen(sCRCTag) > 0)
		bSendCRC = true;
	else bSendCRC = false;

	if(CCI.bUseCompression)
	{
		float dwBeforeSz = (float) Get_FileSize(sFileName);
		float dwAfterSz = 0;

		sprintf_s(sFullFile, sizeof(sFullFile), "%s.sec", sFileName);

		sprintf_s(sFileBuffer, sizeof(sFileBuffer),
			"Compressing file. (%.2f KB)", dwBeforeSz / 1024);
		WriteLogEx(pSockSrvr->icClientID[iClient], sFileBuffer, EVENT_NONE);
		
		if(_strcmpi(CCI.sCompressionMethod, "GZIP") == 0){
			GZip_Deflate_File(sFileName, sFullFile, CCI.dwCompressionLevel);
		}
		else if(_strcmpi(CCI.sCompressionMethod, "LZSS") == 0){
			LZSSCompressFile(sFileName, sFullFile, CCI.dwCompressionLevel);
		}
		else if(_strcmpi(CCI.sCompressionMethod, "LZARI") == 0){
			CompressFile(sFileName, sFullFile);
		}
		else if(_strcmpi(CCI.sCompressionMethod, "RLE") == 0){
			RLEEncodeFile(sFileName, sFullFile);
		}

		dwAfterSz = (float) Get_FileSize(sFullFile);

		sprintf_s(sFileBuffer, sizeof(sFileBuffer),
			"File size reduced from %.2f KB to %.2f KB. (%.2f %% Compression).",
			dwBeforeSz / 1024, dwAfterSz / 1024, (100 - ((dwAfterSz / dwBeforeSz) * 100)));
		WriteLogEx(pSockSrvr->icClientID[iClient], sFileBuffer, EVENT_NONE);
	}
	else{
		strcpy_s(sFullFile, sizeof(sFullFile), sFileName);
	}

	WriteLogEx(pSockSrvr->icClientID[iClient], "Begining file transfer.", EVENT_NONE);
	if(fopen_s(&hSourceHandle, sFullFile, "rb") != 0)
    {
		WriteLogEx(pSockSrvr->icClientID[iClient], "Failed to open export file for binary read.", EVENT_ERROR);
        return false;
    }

    do{
        //if(MyClient.PeekSizeOfSendData() == 0)
        //{
            iBytesRead = fread(sFileBuffer, sizeof(char), FILEBUFFERSZ, hSourceHandle);
			if(bSendCRC)
			{
	            dwDataCRC = PartialCRC(dwDataCRC, sFileBuffer, iBytesRead);
			}
            CCI.SC.Cipher(sFileBuffer, iBytesRead);
			pSockSrvr->SetNextSendDataEx(iClient, sFileBuffer, iBytesRead);
        //}
        //else Sleep(1);
    } while(iBytesRead == FILEBUFFERSZ && pSockSrvr->bcConnected[iClient] && !pSockSrvr->bcDisconnect[iClient]);

	if(bSendCRC)
	{
		dwDataCRC = (dwDataCRC ^ 0xffffffff);
	}
	
	fclose(hSourceHandle);

	DeleteFile(sFullFile);
	DeleteFile(sFileName);

	if(pSockSrvr->bcConnected[iClient] && !pSockSrvr->bcDisconnect[iClient])
	{
		WriteLogEx(pSockSrvr->icClientID[iClient], "File Transfer complete.", EVENT_NONE);
		pSockSrvr->SetNextSendData(iClient, "::EOF");

		if(bSendCRC)
		{
			//Send the data CRC.
			sprintf_s(sFileBuffer, sizeof(sFileBuffer), "::%s->%d", sCRCTag, dwDataCRC);
			pSockSrvr->SetNextSendData(iClient, sFileBuffer);
		}
		return true;
	}
	else return false;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

int ExecuteTransactionSQL(CSockSrvr *pSockSrvr, int iClient)
{
	char sTempFile[MAX_PATH];
    char sSendBuf[SENDBUFSZ + 1];

	int iSendBufSz = 0;
	long lFetchResult = 0;
	int iID = 0;

	char sSQLTemp[2048 + 1];
	char sStatement[2048 + 1];
	char sImportTable[2048 + 1];
	char sRepTable[1024 + 1];
	char sImportDB[2048 + 1];
	int iSz = 0;

	if(gbDebugMode)
	{
		WriteLogEx(pSockSrvr->icClientID[iClient], "Executing transaction SQL.", EVENT_NONE);
	}

	//Loop through all of the statements that need to be executed.
	//This loop is broken when it encounters a script that is supposed to
	//	return data, and in fact does return data.
	while(CCI.rsTrans.FetchEx(&lFetchResult))
	{
		iID = CCI.rsTrans.lColumn(1);

		CCI.rsTrans.sColumnEx(2, sRepTable, sizeof(sRepTable), &iSz);
		if(iSz <= 0)
		{
			strcpy_s(sRepTable, sizeof(sRepTable), "");
		}

		CCI.rsTrans.sColumnEx(3, sStatement, sizeof(sStatement), &iSz);
		if(iSz <= 0)
		{
			strcpy_s(sStatement, sizeof(sStatement), "");
		}

		CCI.rsTrans.sColumnEx(4, CCI.sOnSuccess, sizeof(CCI.sOnSuccess), &iSz);
		if(iSz <= 0)
		{
			strcpy_s(CCI.sOnSuccess, sizeof(CCI.sOnSuccess), "");
		}

		CCI.rsTrans.sColumnEx(5, CCI.sOnFailure, sizeof(CCI.sOnFailure), &iSz);
		if(iSz <= 0)
		{
			strcpy_s(CCI.sOnFailure, sizeof(CCI.sOnFailure), "");
		}

		CCI.rsTrans.sColumnEx(6, sImportDB, sizeof(sImportDB), &iSz);
		if(iSz <= 0)
		{
			strcpy_s(sImportDB, sizeof(sImportDB), "");
		}
		
		CCI.rsTrans.sColumnEx(7, sImportTable, sizeof(sImportTable), &iSz);
		if(iSz <= 0)
		{
			strcpy_s(sImportTable, sizeof(sImportTable), "");
		}

		//If we have an import table name then the script is supposed to return data.
		//	The data returned will be exported and sent to the server for processing.
		if(iSz > 0)
		{
			GetTempFileName(sTempFilesPath, "SEC", 0, sTempFile);

			if(strlen(sRepTable) > 0)
			{
				//Drop the SQLExch_Temp temp table.
				sprintf_s(sSQLTemp, sizeof(sSQLTemp), 
					"IF Exists (SELECT 1 FROM [%s].[%s].[SysObjects]"
					" WHERE [Name] = 'SQLExch_Temp' and [xType] = 'U')"
					" DROP TABLE [%s].[%s].[SQLExch_Temp]",
					gsReplicationDB, gsDefaultDBO, gsReplicationDB, gsDefaultDBO);
				CCI.cCustSQL.ExecuteNonQuery(sSQLTemp);

				//Select the distinct Primary key columns from the Transaction table that
				//	the current statement uses. This will stop the multiple transactions
				//	on the same data row from being replicated multiple times.
				sprintf_s(sSQLTemp, sizeof(sSQLTemp),
					"SELECT Distinct * INTO [%s].[%s].[SQLExch_Temp]"
					" FROM [%s].[%s].[%s]",
					gsReplicationDB, gsDefaultDBO,
					gsReplicationDB, gsDefaultDBO, sRepTable);
				CCI.cCustSQL.ExecuteNonQuery(sSQLTemp);
			}

			int iSQLExpRes = ExportSQLResults(pSockSrvr, iClient, iID, sStatement,
				sImportDB, gsDefaultDBO, sImportTable, sTempFile);

			if(iSQLExpRes == SQL_IMPORT_RESULT_ZEROROWS)
			{
				// The statement was supposed to return data, it didnt... Thats ok!
				// Process the next statement.
				DeleteFile(sTempFile);
				continue;
			}
			else if(iSQLExpRes == SQL_IMPORT_RESULT_OK){

				sprintf_s(sSendBuf, sizeof(sSendBuf), "::TransDataFileSize->%d", Get_FileSize(sTempFile));
				pSockSrvr->SetNextSendData(iClient, sSendBuf);

				sprintf_s(sSendBuf, sizeof(sSendBuf), "::TransDataDatabase->%s", sImportDB);
				pSockSrvr->SetNextSendData(iClient, sSendBuf);

				sprintf_s(sSendBuf, sizeof(sSendBuf), "::TransDataTable->%s", sImportTable);
				pSockSrvr->SetNextSendData(iClient, sSendBuf);

				sprintf_s(sSendBuf, sizeof(sSendBuf), "::TransDataDBO->%s", gsDefaultDBO);
				pSockSrvr->SetNextSendData(iClient, sSendBuf);

				pSockSrvr->SetNextSendData(iClient, "::SendingTransData");

				SendFileData(pSockSrvr, iClient, sTempFile, "TransDataCRC");
				return CMD_OK;
			}
			else if(iSQLExpRes == SQL_IMPORT_RESULT_ERROR){
				// There was an error exporting the SQL data.
				// FIXME: What do we do now, Process the next statement or disconnect?
				//		Need to think over the logic.
				DeleteFile(sTempFile);
				return CMD_ERROR;
			}
		}
		else{
			// The statement was not a statement that is supposed to return data.
			// Just execute it then process the next statement.

			CRecordSet rsTemp;

			if(!CCI.cCustSQL.Execute(sStatement, &rsTemp))
			{
				sprintf_s(sSendBuf, sizeof(sSendBuf), "Execute: Fail on (%s)", sStatement);
				WriteLogEx(pSockSrvr->icClientID[iClient], sSendBuf, EVENT_ERROR);
				return CMD_ERROR;
			}

			if(rsTemp.RowCount > 0)
			{
				sprintf_s(sSendBuf, sizeof(sSendBuf), "Statement affected %d rows.", rsTemp.RowCount);
				WriteLogEx(pSockSrvr->icClientID[iClient], sSendBuf, EVENT_NONE);
			}

			rsTemp.Close();
			continue;
		}
	}

	if(lFetchResult == SQL_NO_DATA_FOUND)
	{
		//Need to delte all transaction from the "SQLExch_Trans" table
		//	where the "SQLExch_Trans.SQLExch_Pending" flag is set to 1.
		//	Any transactions that were added durring the replication process will not
		//	be deleted.
		sprintf_s(sStatement, sizeof(sStatement),
			"DELETE FROM [%s].[%s].[SQLExch_Trans]"
			" WHERE [SQLExch_Pending] = 1",
			gsReplicationDB, gsDefaultDBO);
		CCI.cCustSQL.ExecuteNonQuery(sStatement);

		WriteLogEx(pSockSrvr->icClientID[iClient], "End of statements.", EVENT_NONE);
		pSockSrvr->SetNextSendData(iClient, "::Complete");
		CCI.rsTrans.Close();

		WriteLogEx(pSockSrvr->icClientID[iClient], "The process has been completed successfully.", EVENT_NONE);

		return CMD_OK;
	}
	else{
		sprintf_s(sSendBuf, sizeof(sSendBuf), "SQL Fetch error: %d.", lFetchResult);
		WriteLogEx(pSockSrvr->icClientID[iClient], sSendBuf, EVENT_ERROR);
		return CMD_ERROR;
	}

	return CMD_ERROR;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
#endif
